/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
/** Counter used to generate unique IDs for component definitions. */
let componentDefCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyComponent {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const baseDef = getNgDirectiveDef(componentDefinition);
        const def = {
            ...baseDef,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            template: componentDefinition.template,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            dependencies: baseDef.standalone && componentDefinition.dependencies || null,
            getStandaloneInjector: null,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: `c${componentDefCount++}`,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        initFeatures(def);
        const dependencies = componentDefinition.dependencies;
        def.directiveDefs = extractDefListOrFactory(dependencies, /* pipeDef */ false);
        def.pipeDefs = extractDefListOrFactory(dependencies, /* pipeDef */ true);
        return def;
    });
}
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = extractDefListOrFactory(directives, /* pipeDef */ false);
    def.pipeDefs = extractDefListOrFactory(pipes, /* pipeDef */ true);
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        return res;
    });
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵdefineDirective(directiveDefinition) {
    return noSideEffects(() => {
        const def = getNgDirectiveDef(directiveDefinition);
        initFeatures(def);
        return def;
    });
}
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
/**
 * Checks whether a given Component, Directive or Pipe is marked as standalone.
 * This will return false if passed anything other than a Component, Directive, or Pipe class
 * See this guide for additional information: https://angular.io/guide/standalone-components
 *
 * @param type A reference to a Component, Directive or Pipe.
 * @publicApi
 */
export function isStandalone(type) {
    const def = getComponentDef(type) || getDirectiveDef(type) || getPipeDef(type);
    return def !== null ? def.standalone : false;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
function getNgDirectiveDef(directiveDefinition) {
    const declaredInputs = {};
    return {
        type: directiveDefinition.type,
        providersResolver: null,
        factory: null,
        hostBindings: directiveDefinition.hostBindings || null,
        hostVars: directiveDefinition.hostVars || 0,
        hostAttrs: directiveDefinition.hostAttrs || null,
        contentQueries: directiveDefinition.contentQueries || null,
        declaredInputs,
        exportAs: directiveDefinition.exportAs || null,
        standalone: directiveDefinition.standalone === true,
        selectors: directiveDefinition.selectors || EMPTY_ARRAY,
        viewQuery: directiveDefinition.viewQuery || null,
        features: directiveDefinition.features || null,
        setInput: null,
        findHostDirectiveDefs: null,
        hostDirectives: null,
        inputs: invertObject(directiveDefinition.inputs, declaredInputs),
        outputs: invertObject(directiveDefinition.outputs),
    };
}
function initFeatures(definition) {
    definition.features?.forEach((fn) => fn(definition));
}
function extractDefListOrFactory(dependencies, pipeDef) {
    if (!dependencies) {
        return null;
    }
    const defExtractor = pipeDef ? getPipeDef : extractDirectiveDef;
    return () => (typeof dependencies === 'function' ? dependencies() : dependencies)
        .map(dep => defExtractor(dep))
        .filter(nonNull);
}
//# sourceMappingURL=data:application/json;base64,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