/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { filter, map, take } from 'rxjs/operators';
import { NavigationCancel, NavigationEnd, NavigationError, NavigationSkipped } from '../events';
var NavigationResult;
(function (NavigationResult) {
    NavigationResult[NavigationResult["COMPLETE"] = 0] = "COMPLETE";
    NavigationResult[NavigationResult["FAILED"] = 1] = "FAILED";
    NavigationResult[NavigationResult["REDIRECTING"] = 2] = "REDIRECTING";
})(NavigationResult || (NavigationResult = {}));
/**
 * Performs the given action once the router finishes its next/current navigation.
 *
 * The navigation is considered complete under the following conditions:
 * - `NavigationCancel` event emits and the code is not `NavigationCancellationCode.Redirect` or
 * `NavigationCancellationCode.SupersededByNewNavigation`. In these cases, the
 * redirecting/superseding navigation must finish.
 * - `NavigationError`, `NavigationEnd`, or `NavigationSkipped` event emits
 */
export function afterNextNavigation(router, action) {
    router.events
        .pipe(filter((e) => e instanceof NavigationEnd || e instanceof NavigationCancel ||
        e instanceof NavigationError || e instanceof NavigationSkipped), map(e => {
        if (e instanceof NavigationEnd || e instanceof NavigationSkipped) {
            return NavigationResult.COMPLETE;
        }
        const redirecting = e instanceof NavigationCancel ?
            (e.code === 0 /* NavigationCancellationCode.Redirect */ ||
                e.code === 1 /* NavigationCancellationCode.SupersededByNewNavigation */) :
            false;
        return redirecting ? NavigationResult.REDIRECTING : NavigationResult.FAILED;
    }), filter((result) => result !== NavigationResult.REDIRECTING), take(1))
        .subscribe(() => {
        action();
    });
}
//# sourceMappingURL=data:application/json;base64,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