"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.scheduleTargetAndForget = exports.targetFromTargetString = exports.targetStringFromTarget = exports.fromAsyncIterable = exports.isBuilderOutput = exports.BuilderProgressState = void 0;
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const progress_schema_1 = require("./progress-schema");
Object.defineProperty(exports, "BuilderProgressState", { enumerable: true, get: function () { return progress_schema_1.State; } });
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function isBuilderOutput(obj) {
    if (!obj || typeof obj.then === 'function' || typeof obj.subscribe === 'function') {
        return false;
    }
    if (typeof obj[Symbol.asyncIterator] === 'function') {
        return false;
    }
    return typeof obj.success === 'boolean';
}
exports.isBuilderOutput = isBuilderOutput;
function fromAsyncIterable(iterable) {
    return new rxjs_1.Observable((subscriber) => {
        handleAsyncIterator(subscriber, iterable[Symbol.asyncIterator]()).then(() => subscriber.complete(), (error) => subscriber.error(error));
    });
}
exports.fromAsyncIterable = fromAsyncIterable;
async function handleAsyncIterator(subscriber, iterator) {
    var _a;
    const teardown = new Promise((resolve) => subscriber.add(() => resolve()));
    try {
        while (!subscriber.closed) {
            const result = await Promise.race([teardown, iterator.next()]);
            if (!result || result.done) {
                break;
            }
            subscriber.next(result.value);
        }
    }
    finally {
        await ((_a = iterator.return) === null || _a === void 0 ? void 0 : _a.call(iterator));
    }
}
/**
 * Returns a string of "project:target[:configuration]" for the target object.
 */
function targetStringFromTarget({ project, target, configuration }) {
    return `${project}:${target}${configuration !== undefined ? ':' + configuration : ''}`;
}
exports.targetStringFromTarget = targetStringFromTarget;
/**
 * Return a Target tuple from a string.
 */
function targetFromTargetString(str) {
    const tuple = str.split(/:/, 3);
    if (tuple.length < 2) {
        throw new Error('Invalid target string: ' + JSON.stringify(str));
    }
    return {
        project: tuple[0],
        target: tuple[1],
        ...(tuple[2] !== undefined && { configuration: tuple[2] }),
    };
}
exports.targetFromTargetString = targetFromTargetString;
/**
 * Schedule a target, and forget about its run. This will return an observable of outputs, that
 * as a a teardown will stop the target from running. This means that the Run object this returns
 * should not be shared.
 *
 * The reason this is not part of the Context interface is to keep the Context as normal form as
 * possible. This is really an utility that people would implement in their project.
 *
 * @param context The context of your current execution.
 * @param target The target to schedule.
 * @param overrides Overrides that are used in the target.
 * @param scheduleOptions Additional scheduling options.
 */
function scheduleTargetAndForget(context, target, overrides, scheduleOptions) {
    let resolve = null;
    const promise = new Promise((r) => (resolve = r));
    context.addTeardown(() => promise);
    return (0, rxjs_1.from)(context.scheduleTarget(target, overrides, scheduleOptions)).pipe((0, operators_1.switchMap)((run) => new rxjs_1.Observable((observer) => {
        const subscription = run.output.subscribe(observer);
        return () => {
            subscription.unsubscribe();
            // We can properly ignore the floating promise as it's a "reverse" promise; the teardown
            // is waiting for the resolve.
            // eslint-disable-next-line @typescript-eslint/no-floating-promises
            run.stop().then(resolve);
        };
    })));
}
exports.scheduleTargetAndForget = scheduleTargetAndForget;
//# sourceMappingURL=data:application/json;base64,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