package bmci.esign.backendend.services.impl;

import bmci.esign.backendend.dto.DemandeDto;
import bmci.esign.backendend.dto.DestinataireDto;
import bmci.esign.backendend.dto.request.FilterDto;
import bmci.esign.backendend.dto.response.DashboardChart;
import bmci.esign.backendend.dto.services.IMapClassWithDto;
import bmci.esign.backendend.models.Demande;
import bmci.esign.backendend.models.Destinataire;
import bmci.esign.backendend.models.enums.EStatut;
import bmci.esign.backendend.repositories.DestinataireRepository;
import bmci.esign.backendend.services.DestinataireService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Service
@Slf4j
public class DestinataireServiceImpl implements DestinataireService {

    private final DestinataireRepository destinataireRepository;
    private final IMapClassWithDto<Destinataire, DestinataireDto> destinataireMapping;
    private final IMapClassWithDto<Demande, DemandeDto> demandeMapping;

    public DestinataireServiceImpl(DestinataireRepository destinataireRepository, IMapClassWithDto<Destinataire, DestinataireDto> destinataireMapping, IMapClassWithDto<Demande, DemandeDto> demandeMapping) {
        this.destinataireRepository = destinataireRepository;
        this.destinataireMapping = destinataireMapping;
        this.demandeMapping = demandeMapping;
    }

    @Override
    public DestinataireDto addDestinataire(DestinataireDto destinataireDto) {
        destinataireDto.setStatut(EStatut.WAITING.name());
        Destinataire destinataire = destinataireMapping.convertToEntity(destinataireDto, Destinataire.class);
        destinataire = destinataireRepository.save(destinataire);
        return destinataireMapping.convertToDto(destinataire, DestinataireDto.class);
    }

    @Override
    public List<DestinataireDto> addAll(List<DestinataireDto> destinataireDtos) {
        List<Destinataire> destinataires = destinataireMapping.convertListToListEntity(destinataireDtos, Destinataire.class);
        destinataires = destinataireRepository.saveAll(destinataires);
        return destinataireMapping.convertListToListDto(destinataires, DestinataireDto.class);
    }


    @Override
    public DestinataireDto updateDestinataire(DestinataireDto destinataireDto, Long id) {
         destinataireRepository.save(destinataireMapping.convertToEntity(destinataireDto, Destinataire.class));
        return destinataireDto;
    }

    @Override
    public DestinataireDto getDestinataire(Long id) {
        return destinataireMapping.convertToDto(destinataireRepository.findById(id).orElse(null), DestinataireDto.class);
    }

    @Override
    public Destinataire findDestinataire(Long id) {
        return destinataireRepository.findById(id).orElse(null);
    }

    @Override
    public List<DestinataireDto> findAllByDemandeId(Long id) {
        return destinataireMapping.convertListToListDto(destinataireRepository.findAllByDemandeId(id), DestinataireDto.class);
    }

    @Override
    @Transactional(readOnly = true)
    public Page<DemandeDto> findAllDemandesToSigne(FilterDto destinataireFilterDto) {
        log.info("PROCESSING<>{findAllDemandesToSigne}... REQUEST: {}", destinataireFilterDto);
        try {
            int page = destinataireFilterDto.getPage();
            int size = destinataireFilterDto.getSize();
            Pageable pageable = PageRequest.of(page, size);
            Specification<Destinataire> specification = buildSpecification(destinataireFilterDto);
            Page<Destinataire> destinataires= destinataireRepository.findAll(specification, pageable);
            Page<Destinataire> destinatairePage =  destinataires.map(destinataire -> {
                destinataire.getDemande().getDestinataires().get(0).setPositions(new ArrayList<>());
              return destinataire;
            });
            return destinatairePage.map(this::mapToResponseDto);
        } catch (Exception e) {
            log.error("EXCEPTION<>findAllDemandesToSigne: {}", e.getMessage(), e);
            return Page.empty();
        }
    }

    private Specification<Destinataire> buildSpecification(FilterDto dto) {
        return (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (dto.getCreatedDate() != null) {
                log.info("creation_date: {}",dto.getCreatedDate());
                predicates.add(criteriaBuilder.equal(root.get("creation_date"), dto.getCreatedDate()));
            }
            if(dto.getPriority() != null){
                log.info("priority: {}",dto.getPriority());
                predicates.add(criteriaBuilder.equal(root.get("demande").get("priority"), dto.getPriority()));
            }
            predicates.add(criteriaBuilder.equal(root.get("statut"), EStatut.WAITING));
            predicates.add(criteriaBuilder.equal(root.get("email"), dto.getEmail()));

            return criteriaBuilder.and(predicates.toArray(new Predicate[0]));
        };
    }

    private DemandeDto mapToResponseDto(Destinataire destinataire) {
        return demandeMapping.convertToDto(destinataire.getDemande(), DemandeDto.class);
    }

    @Override
    @Transactional(readOnly = true)
    public List<DashboardChart> loadSignDmChart(String email) {
        log.info("PROCESSING<Destinataire>{loadSignDmChart}... EMAIL: {}", email);
        try {
            if(email.isEmpty()){
                return null;
            }
            return destinataireMapping.mappingData(destinataireRepository.loadSignDm(email));
        }catch (Exception e){
            log.error("EXCEPTION<>loadSignaturesChart: {}", e.getMessage(), e);
            return Collections.emptyList();
        }
    }
}
